//******************************************************************************
//   MSP430x26x Demo - SMBus Master reads from MSP430 SMBus Slave
//
//   Description: This demo connects two MSP430's via the I2C/SMBus. The master 
//   transmits one byte and reads 2. This is the master code. The slave 
//   code is called msp430x26x_SMB_SLV_timeout.c. The data from the slave transmitter 
//   begins at 1 and is incremented with each transfer. The UCB0RXIFG interrupt 
//   is used to know when new data has been received.
//   TimerA0 is set to detect timeout period of 25 ms and issues a STOP,
//   when that happens. P1.0(LED) is toggled on a timeout
//   ACLK = n/a, MCLK = SMCLK = BRCLK = DCO ~ 1.1MHz
//   //* USCI Required *//
//   
//                                 /|\  /|\
//                 MSP430F2619     10k  10k    MSP430F2619
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.1|<-|----+>|P3.1         P1.0|-->LED
//             |                 |  |      |                 |
//             |             P3.2|<-+----->|P3.2             |
//             |                 |         |                 |
//             |             P2.0|<--------|P4.7             |
//
//
//  H. Grewal
//  Texas Instruments Inc.
//  July 2009
//  Built with IAR Embedded Workbench Version: 4.11B
//******************************************************************************

#include  <msp430x26x.h>

typedef enum {
             SMBS_MISC               = (unsigned char) 0x48,
             SMBS_DEVICE_DFLT        = (unsigned char) 0x61,
             SMBS_ALERT_RESPONSE     = (unsigned char) 0x0C,
             SMBS_HOST               = (unsigned char) 0x08,
             SMBS_10BIT_ADD          = (unsigned char) 0x78,
             SMBS_DFLT_ADD           = (unsigned char) 0x37,
             SMBS_ACCESS_HOST        = (unsigned char) 0x28,             
             SMBS_CBUS               = (unsigned char) 0x01,
             SMBS_GCAL               = (unsigned char) 0x00
             }SMBS_Address_t;               
/*---------------------------------------------------------------------------*/

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add);  

unsigned char TXByteCtr;
unsigned char RXByteCtr;

unsigned char TXData = 0;
unsigned char RXData = 0;
unsigned char TestData = 0;


void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  P4OUT = 0;
  P4DIR =  0xFF;                            // P4.7 to reset Slave
  P3SEL |= 0x06;                            // Select I2C pins
  P1DIR |= 0xFF;                            // LED output
                                            // unused ports to output
  P2DIR =  0xFF;
  P2OUT = 0;
  P3DIR =  0xFF;
  
  P5DIR =  0xFF;
  P6DIR =  0xFF;                                      
  P1OUT = 0;                                // Clear P1.0
  TACTL = TASSEL_2;                         // SMCLK
  CCR0 = 27500;                             // Detect 25 ms Timeout, 27500
  CCTL0 = CCIE;                             // Enable Timeout Interrupt
  SMBS_Init (SMBS_MISC,0x48);               // Slave address configured as 48h
    
  __enable_interrupt();                     // GIE/Interrupts enabled
  
  while(1)
  {
   TXByteCtr = 1;                           // 1 byte TX
   RXByteCtr = 2;                           // 2 byte RX
   while(UCB0CTL1 & UCTXSTP);               // Stop condition sent?
   TACTL |= TACLR + MC_1;                   // Upmode, Go
   UCB0CTL1 |= UCTR + UCTXSTT;              // I2C start condition, TX
   __bis_SR_register(LPM0_bits);            // Enter LPM0 w/ interrupts

  }

}

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add)
{
  UCB0CTL1 |= UCSWRST;                      // Enable SW reset
  UCB0CTL0 = UCMST + UCMODE_3 + UCSYNC;     // I2C Master, synchronous mode
  UCB0CTL1 = UCSSEL_2 + UCSWRST;            // Use SMCLK, keep SW reset
  UCB0BR0 = 12;                             // fSCL = SMCLK/12 = ~100kHz
  UCB0BR1 = 0;
  if (Add_Param == SMBS_MISC)
  {
      UCB0I2CSA = Slave_Add;                // Slave Address is 048h
  } 
  else
      UCB0I2CSA = (unsigned char) Add_Param ;
  UCB0CTL1 &= ~UCSWRST;                     // Clear SW reset, resume operation
  IE2 |= UCB0TXIE+UCB0RXIE;                 // Enable TX,RX interrupt 
  UCB0I2CIE |= UCNACKIE;                    // Enable NACK Interrupt
  
}

// Common ISR for I2C Module
#pragma vector = USCIAB0TX_VECTOR
__interrupt void USCIAB0TX_ISR(void)
{
  if (IFG2 & UCB0RXIFG)                     // If RX interrupt
  {
      TACTL &= ~MC_1;                       // Stop Timer
      RXByteCtr--;                          // Decrement RX byte counter
      
      if (RXByteCtr)                        // Multiple bytes left to receive
      {
        RXData = UCB0RXBUF;              
        TACTL |= TACLR + MC_1;              // Re-Start Timer
      }
      else
      {
        RXData= UCB0RXBUF;                  // Last byte to be received
        __bic_SR_register_on_exit(CPUOFF);  // Exit LPM0
      }

      if (RXByteCtr == 1)                   // Only one byte left? 
        UCB0CTL1 |= UCTXSTP;                // Generate I2C stop condition
                                            // If reading only 1 byte, stop condition
                                            // should have already been sent
  }
  else
  {
      TACTL &= ~MC_1;                       // Stop Timer
      if (TXByteCtr)                        // If there's something to transmit
      {
        UCB0TXBUF = TXData++;               // Load TX buffer
        TXByteCtr--;                        // Decrement TX byte counter
      }
      else                                  // Nothing more to transmit 
      {                                      
        
        UCB0CTL1 &= ~UCTR;                  // Receiver mode
        IFG2 &= ~UCB0TXIFG;                 // Clear USCI_B0 TX int flag
        UCB0CTL1 |= UCTXSTT;                // I2C restart (SMBus protocol)
        /* If you're planning on receiving only one byte
         the stop condition should be sent here
        while(UCB0CTL1 & UCTXSTT);          // Start condition sent?
        UCB0CTL1 |= UCTXSTP;                // Send stop signal immediately (during first byte)*/
      }

      TACTL |= TACLR + MC_1;
  }
     
} 

// USCI_B0 State ISR
#pragma vector = USCIAB0RX_VECTOR
__interrupt void USCIAB0RX_ISR(void)
{
   UCB0CTL1 |= UCTXSTP;                  // Transmit a stop
   UCB0STAT &= ~UCNACKIFG;               // Clear NACK flag
   TACTL &= ~MC_1;                       // Stop Timer
   TACTL |= TACLR;
}



// Timer A0 interrupt service routine
#pragma vector=TIMERA0_VECTOR
__interrupt void Timer_A (void)
{
  P4OUT = 0x80;                             // Reset Slave Signal, P4.7  
  TACTL &= ~MC_1;                           // Stop Timer
  TACTL |= TACLR;
  IFG2 &= ~(UCB0TXIFG + UCB0RXIFG);         // ...transactions, free clock line
  RXData= UCB0RXBUF;                        // RX Dummy data to release line
  UCB0TXBUF = 0;                            // TX Dummy data to release line
  UCB0CTL1 |= UCTXSTP;                      // Send Stop  
  while(UCB0CTL1 & UCTXSTP);                // Stop condition sent?
  P1OUT ^= 0x01;                            // Toggle P1.0/LED
  P4OUT = 0;                                // P4.7 = 0  
  SMBS_Init (SMBS_MISC,0x48);               // Reset Master
  __bic_SR_register_on_exit(CPUOFF);        // Exit LPM0
  
} 




























